/**
* \file: partition.h
*
* \version: $Id:$
*
* \release: $Name:$
*
* \component: automounter
*
* \author: Marko Hoyer / ADIT / SWGII / mhoyer@de.adit-jv.com
*
* \copyright (c) 2010, 2011 Advanced Driver Information Technology.
* This code is developed by Advanced Driver Information Technology.
* Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
* All rights reserved.
*
*
***********************************************************************/
#ifndef AMM_MODEL_PARTITION_H
#define AMM_MODEL_PARTITION_H

#include <stdbool.h>

#include "automounter_types.h"

// forward declarations of the data structures that hold the model data. The definition of the data structure is hidden.
// It can be found in the respective C files. Properties can be accessed via respective set and get functions
typedef struct partition_metadata_t partition_metadata_t;
typedef struct partition_t partition_t;
typedef struct device_t device_t;

// declaration of a call back function which can be used to observe state changes of the partition. It is used while
// processing a umount or remount request to detect when it is done.
typedef void (*partition_state_change_callback_t)(partition_t *part, partition_state_t previous_state, void *data); 

//getter for partition attributes
const char *partition_get_id(partition_t *partition);

const char *partition_get_mountsrc(partition_t *partition);

const char *partition_get_mountfs(partition_t *partition);

const char *partition_get_mountpoint(partition_t *partition);

const char *partition_get_partition_identifier(partition_t *partition);

int partition_get_partition_number(partition_t *partition);

bool partition_is_mounted_writable(partition_t *partition);

error_code_t partition_get_error(partition_t *partition);

partition_state_t partition_get_state(partition_t *partition);

const char* partition_get_state_str(partition_state_t state);

partition_unsupported_reason_t partition_get_unsupported_reason(partition_t *partition);

const char* partition_get_unsupported_reason_str(partition_t *partition);

//getter to get a reference to metadata of the partition. Respective get functions defined in partition_metadata.h
//can be used to read out the individual attributes
partition_metadata_t *partition_get_metadata(partition_t *partition);

//getter to get a reference to the parent device. device.h defines get functions to read out the individual attributes.
device_t *partition_get_device(partition_t *partition);


// set or remove a state change monitor
void partition_set_state_change_monitor(partition_t *partition, 
					partition_state_change_callback_t callback_func, void *callback_data);

void partition_remove_state_change_monitor(partition_t *partition);

// set functions to change the partition state in the model. A registered state change callback is called out of these
// functions.
void partition_set_state_mounting(partition_t *partition, const char* mount_point);

void partition_set_state_mounted(partition_t *partition, bool writable);

void partition_return_to_state_mounted_on_err(partition_t* partition, error_code_t err);

void partition_set_state_unmounting(partition_t* partition);

void partition_set_state_unmounted(partition_t* partition);

void partition_set_state_remounting(partition_t *partition);

void partition_set_state_invalid(partition_t *partition);

void partition_set_state_unsupported(partition_t *partition, partition_unsupported_reason_t reason);

void partition_set_state_mount_err(partition_t *partition, error_code_t err);

#endif
